package orthogonalstate;

import java.util.*;

/**
* This class serves as a access point for the whole framework. An FSM object 
* encapsulates a FSM and provides a factory method to create a FSMContext 
* object for this FSM.
*/
public class FSM
{
        Hashtable states = new Hashtable();
        Hashtable events = new Hashtable();

        public static String ROOT="__root__";

        NodeState root = new NodeState(ROOT);

        FSMAction initaction = null;
        
/**
* Initialize the FSM: create a root NodeState.
*/
        public FSM()
        {
                states.put(ROOT, getRoot());
                root.setParent(null);
        }
        
/**
* @return The root NodeState of the fsm
*/
        public NodeState getRoot() { return root; }

/**
* Add a leafstate. 
* @param statename The name of this state
* @param parentstate The name of the parent state. If left null, the state will 
* be added to the root state.
*/
        public void addLeafState(String statename, String parentstate)
        {
                if(parentstate == null)
                {
                        parentstate=ROOT;
                }
                NodeState parent = getRoot();
                try
                {
                        parent = (NodeState)states.get(parentstate);
                }
                catch(Exception e)
                {
                        throw new RuntimeException("state; " + parentstate + 
                                                   " is not a node state");                        
                }
                if(!states.containsKey(statename))
                {
                        LeafState s = new LeafState(statename);
                        s.setParent(parent);
                        parent.addState(s);
                        states.put(statename, s);                        
                }
                else throw new RuntimeException("state; " + statename + 
                                                " already declared");
        }                

/**
* Add a leafstate. 
* @param statename The name of this state
* @param parentstate The name of the parent state. If left null, the state will 
* be added to the root state.
*/
        public void addLeafState(FSMAction entryAction, String statename,
                                 String parentstate)
        {
                if(parentstate == null)
                {
                        parentstate=ROOT;
                }
                NodeState parent = getRoot();
                try
                {
                        parent = (NodeState)states.get(parentstate);                }
                catch(Exception e)
                {
                        throw new RuntimeException("state; " + parentstate + 
                                                   " is not a node state");                        
                }
                if(!states.containsKey(statename))
                {
                        LeafState s = new LeafState(statename);
                        s.setStateEntryAction(entryAction);
                        s.setParent(parent);
                        parent.addState(s);
                        states.put(statename, s);
                }
                else throw new RuntimeException("state; " + statename + 
                                                " already declared");
        }                

/**
* Add a leafstate. 
* @param statename The name of this state
* @param exitAction The action that is executed upon state-exit
* @param parentstate The name of the parent state. If left null, the state will 
* be added to the root state.
*/
        public void addLeafState(String statename, FSMAction exitAction, 
                                 String parentstate)
        {
                if(parentstate == null)
                {
                        parentstate=ROOT;
                }
                NodeState parent = getRoot();
                try
                {
                        parent = (NodeState)states.get(parentstate);                }
                catch(Exception e)
                {
                        throw new RuntimeException("state; " + parentstate + 
                                                   " is not a node state");                        
                }
                if(!states.containsKey(statename))
                {
                        LeafState s = new LeafState(statename);
                        s.setStateExitAction(exitAction);
                        s.setParent(parent);
                        parent.addState(s);
                        states.put(statename, s);
                }
                else throw new RuntimeException("state; " + statename + 
                                                " already declared");
        }                

/**
* Add a leafstate. 
* FSM uses the State class to create a new State object
* @param entryAction The action that is executed upon state-entry
* @param statename The name of the state
* @param exitAction The action that is executed upon state-exit
* @param parentstate The name of the parent state. If left null, the state will 
* be added to the root state.
*/
        public void addLeafState(FSMAction entryAction, String statename, 
                             FSMAction exitAction, String parentstate)
        {
                if(parentstate == null)
                {
                        parentstate=ROOT;
                }
                NodeState parent = getRoot();
                try
                {
                        parent = (NodeState)states.get(parentstate);                }
                catch(Exception e)
                {
                        throw new RuntimeException("state; " + parentstate + 
                                                   " is not a node state");                        
                }
                if(!states.containsKey(statename))
                {
                        LeafState s = new LeafState(statename);
                        s.setStateEntryAction(entryAction);
                        s.setStateExitAction(exitAction);
                        s.setParent(parent);
                        parent.addState(s);
                        states.put(statename, s);
                }
                else throw new RuntimeException("state; " + statename + 
                                                " already declared");
        }   
        
/**
* Add a nodestate.
* @param statename The name of the state
* @param parentstate The name of the parent state. If left null, the state will 
* be added to the root state.
*/
        public void addNodeState(String statename, String parentstate)
        {
                if(parentstate == null)
                {
                        parentstate=ROOT;
                }
                NodeState parent = getRoot();
                try
                {
                        parent = (NodeState)states.get(parentstate);                }
                catch(Exception e)
                {
                        throw new RuntimeException("state; " + parentstate + 
                                                   " is not a node state");                        
                }
                if(!states.containsKey(statename))
                {
                        NodeState s = new NodeState(statename);
                        s.setParent(parent);
                        parent.addState(s);
                        states.put(statename, s);
                        
                }
                else throw new RuntimeException("state; " + statename + 
                                                " already declared");
        }            

        /**
        * This method can be used to add an event to the FSM.
        * FSM uses the Event class to create a new Event object
        * @param name This is the name of the event.
        */
        public void addEvent(String name)
        {
                if(!events.containsKey(name))
                {
                        events.put(name, new FSMEvent(name));
                }
                else throw new RuntimeException("event; " + name + 
                                                " already declared");
        }

        /**
        * This method creates a transition between the sourcestate and the
        * target state. The method checks whether the given states and the
        * event exist before it creates the transition. If they don't exist
        * a RuntimeException is thrown.
        * @param sourcestate The name of the sourcestate
        * @param eventname The name of the event that triggers the transition
        * @param targetstate the name of the targetstate
        * @param action The action that will be executed when the transition
        * is triggered.
        */
        public void addTransition(String sourcestate, String eventname, 
                                  String targetstate, FSMAction action)
        {
                if(states.containsKey(sourcestate))
                {
                        State s = (State)states.get(sourcestate);
                        if(states.containsKey(targetstate))
                        {
                                State t = (State)states.get(targetstate);
                                if(events.containsKey(eventname))
                                {
                                        FSMEvent e = 
                                             (FSMEvent)events.get(eventname);
                                        if(s instanceof LeafState &&
                                           t instanceof LeafState)
                                             ((LeafState)s).addTransition(
                                                                (LeafState)t,
                                                                e, 
                                                                action);
                                        else if (s instanceof LeafState &&
                                                 t instanceof NodeState)
                                             ((LeafState)s).addTransition(
                                                                (NodeState)t,
                                                                e,  
                                                                action);
                                        else if (s instanceof NodeState &&
                                                 t instanceof LeafState)
                                             ((NodeState)s).addTransition(
                                                                (LeafState)t,
                                                                e, 
                                                                action);
                                        else if (s instanceof NodeState &&
                                                 t instanceof NodeState)
                                             ((NodeState)s).addTransition(
                                                                (NodeState)t,
                                                                e,  
                                                                action);
                                        else
                                                System.out.println(
                                                      "this should not happen");
                                }
                                else throw new RuntimeException("event; " + 
                                      eventname + " not found");
                        }
                        else throw new RuntimeException("state; " + targetstate
                               + " not found");
                }
                else throw new RuntimeException("state; " + sourcestate + 
                                                " not found");
        }
        
/**
* Set the default state for the root state.
*/
        public void setDefaultState(String state)
        {
                setDefaultState(ROOT, state);
        }
        
/**
* Set the default state for a nodestate.
* @param parent The nodestate
* @param state The default state
*/
        public void setDefaultState(String parent, String state)
        {
                try
                {      
                        NodeState p = (NodeState)states.get(parent);
                        State s = (State)states.get(state);
                        p.setDefaultState(s);
                }
                catch(Exception e)
                {
                        e.printStackTrace();
                        throw new RuntimeException("impossible to set " + state 
                                                + " as default for " + parent);
                }
        }
        
/**
* Convenience method.
* @param name The name of the state
* @return a state object for name
*/
        public State getStateWithName(String name)
        {
                return (State)states.get(name);
        }

        /**
        * Sometimes it's necessary to do some initialization before the FSM can
        * be used. For this purpos a initial action can be set. This action is
        * executed when the FSMContext is created. 
        * @param action The initial action.
        */
        public void setInitAction(FSMAction action)
        {
                initaction = action;
        }

        /**
        * This method serves as a factory method to create FSMContexts from
        * the FSM. Also the init action is run (if available).
        * @return A new FSMContext for the FSM.
        */
        public FSMContext createFSMInstance()
        {
                FSMContext fsmc;
                if(initaction != null && getRoot() != null)
                {
                        fsmc =  new FSMContext(getRoot(), initaction);
                }
                else
                {
                        fsmc =  new FSMContext(getRoot());
                }
                return fsmc;
        }
}